/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.urlencoding;

import static org.apache.juneau.TestUtils.*;
import static org.apache.juneau.junit.bct.BctAssertions.*;
import static org.junit.jupiter.api.Assertions.*;

import java.util.*;

import org.apache.juneau.*;
import org.apache.juneau.collections.*;
import org.apache.juneau.parser.*;
import org.junit.jupiter.api.*;

@SuppressWarnings("rawtypes")
class UrlEncodingParser_Test extends TestBase {

	static UrlEncodingParser p = UrlEncodingParser.DEFAULT;
	static BeanSession bs = p.getBeanContext().getSession();

	//====================================================================================================
	// Basic test
	//====================================================================================================
	@Test void a01_basic() throws Exception {

		// Simple string
		// Top level
		var t = "_value=a";
		assertEquals("a", p.parse(t, Object.class));
		assertEquals("a", p.parse(t, String.class));
		t = "_value='a'";
		assertEquals("a", p.parse(t, String.class));
		assertEquals("a", p.parse(t, Object.class));
		t = "_value= 'a' ";
		assertEquals("a", p.parse(t, String.class));

		// 2nd level
		t = "?a=a";
		assertEquals("a", p.parse(t, Map.class).get("a"));

		// Simple map
		// Top level
		t = "?a=b&c=123&d=false&e=true&f=null";
		var m = p.parse(t, Map.class);
		assertEquals("b", m.get("a"));
		assertTrue(m.get("c") instanceof Number);
		assertEquals(123, m.get("c"));
		assertTrue(m.get("d") instanceof Boolean);
		assertEquals(Boolean.FALSE, m.get("d"));
		assertTrue(m.get("e") instanceof Boolean);
		assertEquals(Boolean.TRUE, m.get("e"));
		assertNull(m.get("f"));

		t = "?a=true";
		m = p.parse(t, HashMap.class, String.class, Boolean.class);
		assertTrue(m.get("a") instanceof Boolean);
		assertEquals("true", m.get("a").toString());

		// null
		// Top level
		t = "_value=null";
		assertNull(p.parse(t, Object.class));

		// 2nd level
		t = "?null=null";
		m = p.parse(t, Map.class);
		assertTrue(m.containsKey(null));
		assertNull(m.get(null));

		t = "?null=null";
		m = p.parse(t, Map.class);
		assertTrue(m.containsKey(null));
		assertNull(m.get(null));

		// 3rd level
		t = "?null=(null=null)";
		m = p.parse(t, Map.class);
		assertTrue(((Map)m.get(null)).containsKey(null));
		assertNull(((Map)m.get(null)).get(null));

		// Empty array

		// 2nd level in map
		t = "?x=@()";
		m = p.parse(t, HashMap.class, String.class, List.class);
		assertTrue(m.containsKey("x"));
		assertEmpty(m.get("x"));
		m = (Map)p.parse(t, Object.class);
		assertTrue(m.containsKey("x"));
		assertEmpty(m.get("x"));
		t = "?x=@()";
		m = p.parse(t, HashMap.class, String.class, List.class);
		assertTrue(m.containsKey("x"));
		assertEmpty(m.get("x"));

		// Empty 2 dimensional array
		t = "_value=@(@())";
		var l = (List)p.parse(t, Object.class);
		assertSize(1, l);
		l = (List)l.get(0);
		assertEmpty(l);
		t = "0=@()";
		l = p.parse(t, LinkedList.class, List.class);
		assertSize(1, l);
		l = (List)l.get(0);
		assertEmpty(l);

		// Array containing empty string
		// Top level
		t = "_value=@('')";
		l = (List)p.parse(t, Object.class);
		assertSize(1, l);
		assertEquals("", l.get(0));
		t = "0=''";
		l = p.parse(t, List.class, String.class);
		assertSize(1, l);
		assertEquals("", l.get(0));

		// 2nd level
		t = "?''=@('')";
		m = (Map)p.parse(t, Object.class);
		assertEquals("", ((List)m.get("")).get(0));
		t = "?''=@('')";
		m = p.parse(t, HashMap.class, String.class, List.class);
		assertEquals("", ((List)m.get("")).get(0));

		// Array containing 3 empty strings
		t = "_value=@('','','')";
		l = (List)p.parse(t, Object.class);
		assertSize(3, l);
		assertEquals("", l.get(0));
		assertEquals("", l.get(1));
		assertEquals("", l.get(2));
		t = "0=''&1=''&2=''";
		l = p.parse(t, List.class, Object.class);
		assertSize(3, l);
		assertEquals("", l.get(0));
		assertEquals("", l.get(1));
		assertEquals("", l.get(2));

		// String containing \u0000
		// Top level
		t = "_value='\u0000'";
		assertEquals("\u0000", p.parse(t, Object.class));
		t = "_value='\u0000'";
		assertEquals("\u0000", p.parse(t, String.class));
		assertEquals("\u0000", p.parse(t, Object.class));

		// 2nd level
		t = "?'\u0000'='\u0000'";
		m = (Map)p.parse(t, Object.class);
		assertSize(1, m);
		assertEquals("\u0000", m.get("\u0000"));
		m = p.parse(t, HashMap.class, String.class, Object.class);
		assertSize(1, m);
		assertEquals("\u0000", m.get("\u0000"));

		// Boolean
		// Top level
		t = "_value=false";
		var b = (Boolean)p.parse(t, Object.class);
		assertEquals(Boolean.FALSE, b);
		b = p.parse(t, Boolean.class);
		assertEquals(Boolean.FALSE, b);
		t = "_value=false";
		b = p.parse(t, Boolean.class);
		assertEquals(Boolean.FALSE, b);

		// 2nd level
		t = "?x=false";
		m = (Map)p.parse(t, Object.class);
		assertEquals(Boolean.FALSE, m.get("x"));
		t = "?x=false";
		m = p.parse(t, HashMap.class, String.class, Boolean.class);
		assertEquals(Boolean.FALSE, m.get("x"));

		// Number
		// Top level
		t = "_value=123";
		var i = (Integer)p.parse(t, Object.class);
		assertEquals(123, i.intValue());
		i = p.parse(t, Integer.class);
		assertEquals(123, i.intValue());
		var d = p.parse(t, Double.class);
		assertEquals(123, d.intValue());
		var f = p.parse(t, Float.class);
		assertEquals(123, f.intValue());
		t = "_value=123";
		i = p.parse(t, Integer.class);
		assertEquals(123, i.intValue());

		// 2nd level
		t = "?x=123";
		m = (Map)p.parse(t, Object.class);
		assertEquals(123, ((Integer)m.get("x")).intValue());
		m = p.parse(t, HashMap.class, String.class, Double.class);
		assertEquals(123, ((Double)m.get("x")).intValue());

		// Unencoded chars
		// Top level
		t = "_value=x;/?:@-_.!*'";
		assertEquals("x;/?:@-_.!*'", p.parse(t, Object.class));

		// 2nd level
		t = "?x;/?:@-_.!*'=x;/?:@-_.!*'";
		m = (Map)p.parse(t, Object.class);
		assertEquals("x;/?:@-_.!*'", m.get("x;/?:@-_.!*'"));
		m = p.parse(t, HashMap.class, String.class, Object.class);
		assertEquals("x;/?:@-_.!*'", m.get("x;/?:@-_.!*'"));
		m = p.parse(t, HashMap.class, String.class, String.class);
		assertEquals("x;/?:@-_.!*'", m.get("x;/?:@-_.!*'"));

		// Encoded chars
		// Top level
		assertThrows(ParseException.class, ()->p.parse("_value=x{}|\\^[]`<>#%\"&+", Object.class));
		t = "_value=x%7B%7D%7C%5C%5E%5B%5D%60%3C%3E%23%25%22%26%2B";
		assertEquals("x{}|\\^[]`<>#%\"&+", p.parse(t, Object.class));
		assertEquals("x{}|\\^[]`<>#%\"&+", p.parse(t, String.class));

		// 2nd level
		assertThrows(ParseException.class, ()->p.parse("?x{}|\\^[]`<>#%\"&+=x{}|\\^[]`<>#%\"&+", Object.class));
		t = "?x%7B%7D%7C%5C%5E%5B%5D%60%3C%3E%23%25%22%26%2B=x%7B%7D%7C%5C%5E%5B%5D%60%3C%3E%23%25%22%26%2B";
		m = (Map)p.parse(t, Object.class);
		assertEquals("x{}|\\^[]`<>#%\"&+", m.get("x{}|\\^[]`<>#%\"&+"));

		// Special chars
		// These characters are escaped and not encoded.
		// Top level
		t = "_value='x$,()'";
		assertEquals("x$,()", p.parse(t, Object.class));
		t = "_value='x~~$~~,~~(~~)'";
		assertEquals("x~$~,~(~)", p.parse(t, Object.class));

		// At secondary levels, these characters are escaped and not encoded.
		// 2nd level
		t = "?'x$,()'='x$,()'";
		m = (Map)p.parse(t, Object.class);
		assertEquals("x$,()", m.get("x$,()"));
		t = "?'x~~$~~,~~(~~)'='x~~$~~,~~(~~)'";
		m = (Map)p.parse(t, Object.class);
		assertEquals("x~$~,~(~)", m.get("x~$~,~(~)"));

		// Equals sign
		// Gets encoded at top level, and encoded+escaped at 2nd level.
		// Top level
		t = "_value='x='";
		assertEquals("x=", p.parse(t, Object.class));
		t = "_value='x%3D'";
		assertEquals("x=", p.parse(t, Object.class));

		// 2nd level
		t = "?'x%3D'='x%3D'";
		m = (Map)p.parse(t, Object.class);
		assertEquals("x=", m.get("x="));
		t = "?'x~~%3D'='x~~%3D'";
		m = (Map)p.parse(t, Object.class);
		assertEquals("x~=", m.get("x~="));

		// String starting with parenthesis
		// Top level
		t = "_value='()'";
		assertEquals("()", p.parse(t, Object.class));
		assertEquals("()", p.parse(t, String.class));
		t = "_value='()'";
		assertEquals("()", p.parse(t, Object.class));
		assertEquals("()", p.parse(t, String.class));

		// 2nd level
		t = "?'()'='()'";
		m = (Map)p.parse(t, Object.class);
		assertEquals("()", m.get("()"));
		t = "?'()'='()'";
		m = p.parse(t, HashMap.class, String.class, Object.class);
		assertEquals("()", m.get("()"));

		// String starting with $
		// Top level
		t = "_value=$a";
		assertEquals("$a", p.parse(t, Object.class));
		t = "_value=$a";
		assertEquals("$a", p.parse(t, Object.class));

		// 2nd level
		t = "?$a=$a";
		m = (Map)p.parse(t, Object.class);
		assertEquals("$a", m.get("$a"));
		m = p.parse(t, HashMap.class, String.class, Object.class);
		assertEquals("$a", m.get("$a"));

		// Blank string
		// Top level
		t = "_value=";
		assertEquals("", p.parse(t, Object.class));

		// 2nd level
		t = "?=";
		m = (Map)p.parse(t, Object.class);
		assertEquals("", m.get(""));
		m = p.parse(t, HashMap.class, String.class, Object.class);
		assertEquals("", m.get(""));

		// 3rd level
		t = "?=(=)";
		m = (Map)p.parse(t, Object.class);
		assertEquals("", ((Map)m.get("")).get(""));
		t = "?=(=)";
		m = p.parse(t, HashMap.class, String.class, HashMap.class);
		assertEquals("", ((Map)m.get("")).get(""));

		// Newline character
		// Top level
		t = "_value='%0A'";
		assertEquals("\n", p.parse(t, Object.class));

		// 2nd level
		t = "?'%0A'='%0A'";
		m = (Map)p.parse(t, Object.class);
		assertEquals("\n", m.get("\n"));

		// 3rd level
		t = "?'%0A'=('%0A'='%0A')";
		m = (Map)p.parse(t, Object.class);
		assertEquals("\n", ((Map)m.get("\n")).get("\n"));
	}

	//====================================================================================================
	// Unicode character test
	//====================================================================================================
	@Test void a02_unicodeChars() throws Exception {
		// 2-byte UTF-8 character
		// Top level
		var t = "_value=¢";
		assertEquals("¢", p.parse(t, Object.class));
		assertEquals("¢", p.parse(t, String.class));
		t = "_value=%C2%A2";
		assertEquals("¢", p.parse(t, Object.class));
		assertEquals("¢", p.parse(t, String.class));

		// 2nd level
		t = "?%C2%A2=%C2%A2";
		var m = (Map)p.parse(t, Object.class);
		assertEquals("¢", m.get("¢"));

		// 3rd level
		t = "?%C2%A2=(%C2%A2=%C2%A2)";
		m = (Map)p.parse(t, Object.class);
		assertEquals("¢", ((Map)m.get("¢")).get("¢"));

		// 3-byte UTF-8 character
		// Top level
		t = "_value=€";
		assertEquals("€", p.parse(t, Object.class));
		assertEquals("€", p.parse(t, String.class));
		t = "_value=%E2%82%AC";
		assertEquals("€", p.parse(t, Object.class));
		assertEquals("€", p.parse(t, String.class));

		// 2nd level
		t = "?%E2%82%AC=%E2%82%AC";
		m = (Map)p.parse(t, Object.class);
		assertEquals("€", m.get("€"));

		// 3rd level
		t = "?%E2%82%AC=(%E2%82%AC=%E2%82%AC)";
		m = (Map)p.parse(t, Object.class);
		assertEquals("€", ((Map)m.get("€")).get("€"));

		// 4-byte UTF-8 character
		// Top level
		t = "_value=𤭢";
		assertEquals("𤭢", p.parse(t, Object.class));
		assertEquals("𤭢", p.parse(t, String.class));
		t = "_value=%F0%A4%AD%A2";
		assertEquals("𤭢", p.parse(t, Object.class));
		assertEquals("𤭢", p.parse(t, String.class));

		// 2nd level
		t = "?%F0%A4%AD%A2=%F0%A4%AD%A2";
		m = (Map)p.parse(t, Object.class);
		assertEquals("𤭢", m.get("𤭢"));

		// 3rd level
		t = "?%F0%A4%AD%A2=(%F0%A4%AD%A2=%F0%A4%AD%A2)";
		m = (Map)p.parse(t, Object.class);
		assertEquals("𤭢", ((Map)m.get("𤭢")).get("𤭢"));
	}

	//====================================================================================================
	// Test simple bean
	//====================================================================================================
	@Test void a03_simpleBean() throws Exception {
		var p2 = UrlEncodingParser.DEFAULT;
		var s = "?f1=foo&f2=123";
		var t = p2.parse(s, A.class);
		assertEquals("foo", t.f1);
		assertEquals(123, t.f2);
	}

	public static class A {
		public String f1;
		public int f2;
	}

	//====================================================================================================
	// Test URL-encoded strings with no-value parameters.
	//====================================================================================================
	@Test void a04_noValues() throws Exception {
		var p2 = UrlEncodingParser.DEFAULT;
		var s = "?f1";
		var m = p2.parse(s, JsonMap.class);
		assertTrue(m.containsKey("f1"));
		assertNull(m.get("f1"));
		s = "?f1=f2&f3";
		m = p2.parse(s, JsonMap.class);
		assertEquals("f2", m.get("f1"));
		assertTrue(m.containsKey("f3"));
		assertNull(m.get("f3"));
	}

	//====================================================================================================
	// Test comma-delimited list parameters.
	//====================================================================================================
	@Test void a05_commaDelimitedLists() throws Exception {
		var p2 = UrlEncodingParser.DEFAULT;
		var s = "?f1=1,2,3&f2=a,b,c&f3=true,false&f4=&f5";
		var c = p2.parse(s, C.class);
		assertBean(c, "f1,f2,f3,f4", "[1,2,3],[a,b,c],[true,false],[]");
	}

	public static class C {
		public int[] f1;
		public String[] f2;
		public boolean[] f3;
		public String[] f4;
		public String[] f5;
	}

	//====================================================================================================
	// Test comma-delimited list parameters with special characters.
	//====================================================================================================
	@Test void a06_commaDelimitedListsWithSpecialChars() throws Exception {
		var p2 = UrlEncodingParser.DEFAULT;

		// In the string below, the ~ character should not be interpreted as an escape.
		var s = "?f1=a~b,a~b";
		var c = p2.parse(s, C1.class);
		assertBean(c, "f1", "[a~b,a~b]");

		s = "?f1=@(a~b,a~b)";
		c = p2.parse(s, C1.class);
		assertBean(c, "f1", "[a~b,a~b]");

		s = "?f1=@('a~b','a~b')";
		c = p2.parse(s, C1.class);
		assertBean(c, "f1", "[a~b,a~b]");

		s = "?f1=@('a~b','a~b')";
		c = p2.parse(s, C1.class);
		assertBean(c, "f1", "[a~b,a~b]");

		s = "?f1=@('a~b','a~b')";
		c = p2.parse(s, C1.class);
		assertBean(c, "f1", "[a~b,a~b]");

		s = "?f1=~~,~~";
		c = p2.parse(s, C1.class);
		assertJson("{f1:['~','~']}", c);

		s = "?f1=@(~~,~~)";
		c = p2.parse(s, C1.class);
		assertJson("{f1:['~','~']}", c);

		s = "?f1=@(~~~~~~,~~~~~~)";
		c = p2.parse(s, C1.class);
		assertJson("{f1:['~~~','~~~']}", c);

		s = "?f1=@('~~~~~~','~~~~~~')";
		c = p2.parse(s, C1.class);
		assertJson("{f1:['~~~','~~~']}", c);

		// The ~ should be treated as an escape if followed by any of the following characters:  '~
		s = "?f1=~'~~,~'~~";
		c = p2.parse(s, C1.class);
		assertJson("{f1:['\\'~','\\'~']}", c);

		s = "?f1=@(~'~~,~'~~)";
		c = p2.parse(s, C1.class);
		assertJson("{f1:['\\'~','\\'~']}", c);

		s = "?f1=@('~'~~','~'~~')";
		c = p2.parse(s, C1.class);
		assertJson("{f1:['\\'~','\\'~']}", c);

		s = "?a~b=a~b";
		var m = p2.parse(s, JsonMap.class);
		assertEquals("{'a~b':'a~b'}", m.toString());

		s = "?'a~b'='a~b'";
		m = p2.parse(s, JsonMap.class);
		assertEquals("{'a~b':'a~b'}", m.toString());

		s = "?~~=~~";
		m = p2.parse(s, JsonMap.class);
		assertEquals("{'~':'~'}", m.toString());

		s = "?'~~'='~~'";
		m = p2.parse(s, JsonMap.class);
		assertEquals("{'~':'~'}", m.toString());

		s = "?~~~~~~=~~~~~~";
		m = p2.parse(s, JsonMap.class);
		assertEquals("{'~~~':'~~~'}", m.toString());

		s = "?'~~~~~~'='~~~~~~'";
		m = p2.parse(s, JsonMap.class);
		assertEquals("{'~~~':'~~~'}", m.toString());
	}

	public static class C1 {
		public String[] f1;
	}

	//====================================================================================================
	// Test comma-delimited list parameters.
	//====================================================================================================
	@Test void a07_whitespace() throws Exception {
		var p2 = UrlEncodingParser.DEFAULT;
		var s = "?f1=foo\n\t&f2=bar\n\t";
		var m = p2.parse(s, JsonMap.class);
		assertEquals("{f1:'foo',f2:'bar'}", m.toString());

		s = "?f1='\n\t'&f2='\n\t'";
		m = p2.parse(s, JsonMap.class);
		assertEquals("\n\t", m.getString("f1"));
		assertEquals("\n\t", m.getString("f2"));

		s = "?f1='\n\t'\n\t&f2='\n\t'\n\t";
		m = p2.parse(s, JsonMap.class);
		assertEquals("\n\t", m.getString("f1"));
		assertEquals("\n\t", m.getString("f2"));
		assertEquals("{f1:'\\n\\t',f2:'\\n\\t'}", m.toString());  // Note that JsonSerializer escapes newlines and tabs.

		s = "?f1='\n\t'\n\t&f2='\n\t'\n\t";
		m = p2.parse(s, JsonMap.class);
		assertEquals("\n\t", m.getString("f1"));
		assertEquals("\n\t", m.getString("f2"));
		assertEquals("{f1:'\\n\\t',f2:'\\n\\t'}", m.toString());  // Note that JsonSerializer escapes newlines and tabs.

		s = "?f1=(\n\tf1a=a,\n\tf1b=b\n\t)\n\t&f2=(\n\tf2a=a,\n\tf2b=b\n\t)\n\t";
		m = p2.parse(s, JsonMap.class);
		assertEquals("{f1:{f1a:'a',f1b:'b'},f2:{f2a:'a',f2b:'b'}}", m.toString());  // Note that JsonSerializer escapes newlines and tabs.
		var d = p2.parse(s, D.class);
		assertBean(d, "f1{f1a,f1b},f2{f2a,f2b}", "{a,b},{a,b}");

		s = "?f1=(\n\tf1a='\n\t',\n\tf1b='\n\t'\n\t)\n\t&f2=(\n\tf2a='\n\t',\n\tf2b='\n\t'\n\t)\n\t";
		m = p2.parse(s, JsonMap.class);
		assertEquals("{f1:{f1a:'\\n\\t',f1b:'\\n\\t'},f2:{f2a:'\\n\\t',f2b:'\\n\\t'}}", m.toString());  // Note that JsonSerializer escapes newlines and tabs.
		d = p2.parse(s, D.class);
		assertBean(d, "f1{f1a,f1b},f2{f2a,f2b}", "{\n\t,\n\t},{\n\t,\n\t}");

		s = "?f1=@(\n\tfoo,\n\tbar\n\t)\n\t&f2=@(\n\tfoo,\n\tbar\n\t)\n\t";
		m = p2.parse(s, JsonMap.class);
		assertEquals("{f1:['foo','bar'],f2:['foo','bar']}", m.toString());  // Note that JsonSerializer escapes newlines and tabs.

		s = "f1=a,\n\tb,\n\tc\n\t&f2=1,\n\t2,\n\t3\n\t&f3=true,\n\tfalse\n\t";
		var e = p2.parse(s, E.class);
		assertBean(e, "f1,f2,f3", "[a,b,c],[1,2,3],[true,false]");

		s = "f1=a%2C%0D%0Ab%2C%0D%0Ac%0D%0A&f2=1%2C%0D%0A2%2C%0D%0A3%0D%0A&f3=true%2C%0D%0Afalse%0D%0A";
		e = p2.parse(s, E.class);
		assertBean(e, "f1,f2,f3", "[a,b,c],[1,2,3],[true,false]");
	}

	public static class D {
		public D1 f1;
		public D2 f2;
	}

	public static class D1 {
		public String f1a, f1b;
	}

	public static class D2 {
		public String f2a, f2b;
	}

	public static class E {
		public String[] f1;
		public int[] f2;
		public boolean[] f3;
	}

	//====================================================================================================
	// Multi-part parameters on beans via URLENC_expandedParams
	//====================================================================================================
	@Test void a08_multiPartParametersOnBeansViaProperty() throws Exception {
		var p2 = UrlEncodingParser.create().expandedParams().build();
		var in = """
			f01=a&f01=b\
			&f02=c&f02=d\
			&f03=1&f03=2\
			&f04=3&f04=4\
			&f05=@(e,f)&f05=@(g,h)\
			&f06=@(i,j)&f06=@(k,l)\
			&f07=(a=a,b=1,c=true)&f07=(a=b,b=2,c=false)\
			&f08=(a=a,b=1,c=true)&f08=(a=b,b=2,c=false)\
			&f09=@((a=a,b=1,c=true))&f09=@((a=b,b=2,c=false))\
			&f10=@((a=a,b=1,c=true))&f10=@((a=b,b=2,c=false))\
			&f11=a&f11=b\
			&f12=c&f12=d\
			&f13=1&f13=2\
			&f14=3&f14=4\
			&f15=@(e,f)&f15=@(g,h)\
			&f16=@(i,j)&f16=@(k,l)\
			&f17=(a=a,b=1,c=true)&f17=(a=b,b=2,c=false)\
			&f18=(a=a,b=1,c=true)&f18=(a=b,b=2,c=false)\
			&f19=@((a=a,b=1,c=true))&f19=@((a=b,b=2,c=false))\
			&f20=@((a=a,b=1,c=true))&f20=@((a=b,b=2,c=false))""";

		var t = p2.parse(in, DTOs.B.class);
		assertBean(t,
			"f01,f02,f03,f04,f05,f06,f07{#{a,b,c}},f08{#{a,b,c}},f09{#{#{a,b,c}}},f10{#{#{a,b,c}}},f11,f12,f13,f14,f15,f16,f17{#{a,b,c}},f18{#{a,b,c}},f19{#{#{a,b,c}}},f20{#{#{a,b,c}}}",
			"[a,b],[c,d],[1,2],[3,4],[[e,f],[g,h]],[[i,j],[k,l]],{[{a,1,true},{b,2,false}]},{[{a,1,true},{b,2,false}]},{[{[{a,1,true}]},{[{b,2,false}]}]},{[{[{a,1,true}]},{[{b,2,false}]}]},[a,b],[c,d],[1,2],[3,4],[[e,f],[g,h]],[[i,j],[k,l]],{[{a,1,true},{b,2,false}]},{[{a,1,true},{b,2,false}]},{[{[{a,1,true}]},{[{b,2,false}]}]},{[{[{a,1,true}]},{[{b,2,false}]}]}");
	}

	@Test void a09_multiPartParametersOnBeansViaProperty_usingConfig() throws Exception {
		var p2 = UrlEncodingParser.create().expandedParams().applyAnnotations(DTOs2.Annotations.class).build();
		var in = """
			f01=a&f01=b\
			&f02=c&f02=d\
			&f03=1&f03=2\
			&f04=3&f04=4\
			&f05=@(e,f)&f05=@(g,h)\
			&f06=@(i,j)&f06=@(k,l)\
			&f07=(a=a,b=1,c=true)&f07=(a=b,b=2,c=false)\
			&f08=(a=a,b=1,c=true)&f08=(a=b,b=2,c=false)\
			&f09=@((a=a,b=1,c=true))&f09=@((a=b,b=2,c=false))\
			&f10=@((a=a,b=1,c=true))&f10=@((a=b,b=2,c=false))\
			&f11=a&f11=b\
			&f12=c&f12=d\
			&f13=1&f13=2\
			&f14=3&f14=4\
			&f15=@(e,f)&f15=@(g,h)\
			&f16=@(i,j)&f16=@(k,l)\
			&f17=(a=a,b=1,c=true)&f17=(a=b,b=2,c=false)\
			&f18=(a=a,b=1,c=true)&f18=(a=b,b=2,c=false)\
			&f19=@((a=a,b=1,c=true))&f19=@((a=b,b=2,c=false))\
			&f20=@((a=a,b=1,c=true))&f20=@((a=b,b=2,c=false))""";

		assertBean(p2.parse(in, DTOs2.B.class),
			"f01,f02,f03,f04,f05,f06,f07{#{a,b,c}},f08{#{a,b,c}},f09{#{#{a,b,c}}},f10{#{#{a,b,c}}},f11,f12,f13,f14,f15,f16,f17{#{a,b,c}},f18{#{a,b,c}},f19{#{#{a,b,c}}},f20{#{#{a,b,c}}}",
			"[a,b],[c,d],[1,2],[3,4],[[e,f],[g,h]],[[i,j],[k,l]],{[{a,1,true},{b,2,false}]},{[{a,1,true},{b,2,false}]},{[{[{a,1,true}]},{[{b,2,false}]}]},{[{[{a,1,true}]},{[{b,2,false}]}]},[a,b],[c,d],[1,2],[3,4],[[e,f],[g,h]],[[i,j],[k,l]],{[{a,1,true},{b,2,false}]},{[{a,1,true},{b,2,false}]},{[{[{a,1,true}]},{[{b,2,false}]}]},{[{[{a,1,true}]},{[{b,2,false}]}]}");
	}

	//====================================================================================================
	// Multi-part parameters on beans via @UrlEncoding.expandedParams on class
	//====================================================================================================
	@Test void a10_multiPartParametersOnBeansViaAnnotationOnClass() throws Exception {
		var p2 = UrlEncodingParser.DEFAULT;
		var in = """
			f01=a&f01=b\
			&f02=c&f02=d\
			&f03=1&f03=2\
			&f04=3&f04=4\
			&f05=@(e,f)&f05=@(g,h)\
			&f06=@(i,j)&f06=@(k,l)\
			&f07=(a=a,b=1,c=true)&f07=(a=b,b=2,c=false)\
			&f08=(a=a,b=1,c=true)&f08=(a=b,b=2,c=false)\
			&f09=@((a=a,b=1,c=true))&f09=@((a=b,b=2,c=false))\
			&f10=@((a=a,b=1,c=true))&f10=@((a=b,b=2,c=false))\
			&f11=a&f11=b\
			&f12=c&f12=d\
			&f13=1&f13=2\
			&f14=3&f14=4\
			&f15=@(e,f)&f15=@(g,h)\
			&f16=@(i,j)&f16=@(k,l)\
			&f17=(a=a,b=1,c=true)&f17=(a=b,b=2,c=false)\
			&f18=(a=a,b=1,c=true)&f18=(a=b,b=2,c=false)\
			&f19=@((a=a,b=1,c=true))&f19=@((a=b,b=2,c=false))\
			&f20=@((a=a,b=1,c=true))&f20=@((a=b,b=2,c=false))""";

		assertBean(p2.parse(in, DTOs.C.class),
			"f01,f02,f03,f04,f05,f06,f07{#{a,b,c}},f08{#{a,b,c}},f09{#{#{a,b,c}}},f10{#{#{a,b,c}}},f11,f12,f13,f14,f15,f16,f17{#{a,b,c}},f18{#{a,b,c}},f19{#{#{a,b,c}}},f20{#{#{a,b,c}}}",
			"[a,b],[c,d],[1,2],[3,4],[[e,f],[g,h]],[[i,j],[k,l]],{[{a,1,true},{b,2,false}]},{[{a,1,true},{b,2,false}]},{[{[{a,1,true}]},{[{b,2,false}]}]},{[{[{a,1,true}]},{[{b,2,false}]}]},[a,b],[c,d],[1,2],[3,4],[[e,f],[g,h]],[[i,j],[k,l]],{[{a,1,true},{b,2,false}]},{[{a,1,true},{b,2,false}]},{[{[{a,1,true}]},{[{b,2,false}]}]},{[{[{a,1,true}]},{[{b,2,false}]}]}");
	}

	@Test void a11_multiPartParametersOnBeansViaAnnotationOnClass_usingConfig() throws Exception {
		var p2 = UrlEncodingParser.DEFAULT.copy().applyAnnotations(DTOs2.Annotations.class).build();
		var in = """
			f01=a&f01=b\
			&f02=c&f02=d\
			&f03=1&f03=2\
			&f04=3&f04=4\
			&f05=@(e,f)&f05=@(g,h)\
			&f06=@(i,j)&f06=@(k,l)\
			&f07=(a=a,b=1,c=true)&f07=(a=b,b=2,c=false)\
			&f08=(a=a,b=1,c=true)&f08=(a=b,b=2,c=false)\
			&f09=@((a=a,b=1,c=true))&f09=@((a=b,b=2,c=false))\
			&f10=@((a=a,b=1,c=true))&f10=@((a=b,b=2,c=false))\
			&f11=a&f11=b\
			&f12=c&f12=d\
			&f13=1&f13=2\
			&f14=3&f14=4\
			&f15=@(e,f)&f15=@(g,h)\
			&f16=@(i,j)&f16=@(k,l)\
			&f17=(a=a,b=1,c=true)&f17=(a=b,b=2,c=false)\
			&f18=(a=a,b=1,c=true)&f18=(a=b,b=2,c=false)\
			&f19=@((a=a,b=1,c=true))&f19=@((a=b,b=2,c=false))\
			&f20=@((a=a,b=1,c=true))&f20=@((a=b,b=2,c=false))""";

		assertBean(p2.parse(in, DTOs2.C.class),
			"f01,f02,f03,f04,f05,f06,f07{#{a,b,c}},f08{#{a,b,c}},f09{#{#{a,b,c}}},f10{#{#{a,b,c}}},f11,f12,f13,f14,f15,f16,f17{#{a,b,c}},f18{#{a,b,c}},f19{#{#{a,b,c}}},f20{#{#{a,b,c}}}",
			"[a,b],[c,d],[1,2],[3,4],[[e,f],[g,h]],[[i,j],[k,l]],{[{a,1,true},{b,2,false}]},{[{a,1,true},{b,2,false}]},{[{[{a,1,true}]},{[{b,2,false}]}]},{[{[{a,1,true}]},{[{b,2,false}]}]},[a,b],[c,d],[1,2],[3,4],[[e,f],[g,h]],[[i,j],[k,l]],{[{a,1,true},{b,2,false}]},{[{a,1,true},{b,2,false}]},{[{[{a,1,true}]},{[{b,2,false}]}]},{[{[{a,1,true}]},{[{b,2,false}]}]}");
	}
}