/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.data.pipeline.mysql.datasource;

import org.apache.shardingsphere.data.pipeline.spi.JdbcQueryPropertiesExtension;
import org.apache.shardingsphere.database.connector.core.spi.DatabaseTypedSPILoader;
import org.apache.shardingsphere.database.connector.core.type.DatabaseType;
import org.apache.shardingsphere.infra.spi.type.typed.TypedSPILoader;
import org.junit.jupiter.api.Test;

import java.util.Optional;
import java.util.Properties;

import static org.hamcrest.Matchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.isA;
import static org.junit.jupiter.api.Assertions.assertTrue;

class MySQLJdbcQueryPropertiesExtensionTest {
    
    @Test
    void assertExtendQueryProperties() {
        Optional<JdbcQueryPropertiesExtension> extension = DatabaseTypedSPILoader.findService(JdbcQueryPropertiesExtension.class, TypedSPILoader.getService(DatabaseType.class, "MySQL"));
        assertTrue(extension.isPresent());
        assertExtension(extension.get());
        Properties props = new Properties();
        extension.get().extendQueryProperties(props);
        assertQueryProperties(props, "600");
        props = new Properties();
        props.setProperty("netTimeoutForStreamingResults", "3600");
        extension.get().extendQueryProperties(props);
        assertQueryProperties(props, "3600");
    }
    
    private void assertExtension(final JdbcQueryPropertiesExtension actual) {
        assertThat(actual, isA(MySQLJdbcQueryPropertiesExtension.class));
        assertThat(actual.getType(), is(TypedSPILoader.getService(DatabaseType.class, "MySQL")));
    }
    
    private void assertQueryProperties(final Properties actual, final String expectedNetTimeoutForStreamingResults) {
        assertThat(actual.size(), is(7));
        assertThat(actual.getProperty("useServerPrepStmts"), is(Boolean.FALSE.toString()));
        assertThat(actual.getProperty("rewriteBatchedStatements"), is(Boolean.TRUE.toString()));
        assertThat(actual.getProperty("yearIsDateType"), is(Boolean.FALSE.toString()));
        assertThat(actual.getProperty("zeroDateTimeBehavior"), is("convertToNull"));
        assertThat(actual.getProperty("noDatetimeStringSync"), is(Boolean.TRUE.toString()));
        assertThat(actual.getProperty("jdbcCompliantTruncation"), is(Boolean.FALSE.toString()));
        assertThat(actual.getProperty("netTimeoutForStreamingResults"), is(expectedNetTimeoutForStreamingResults));
    }
}
